implementation module _SystemDynamic


import StdEnv

/*	----------------------------------------------------------------------------
	TypeCode
-----------------------------------------------------------------------------  */

:: TypeCode
	=	TypeScheme !Int TypeCode		// Int = # variables
	|	TypeVar !Int					// Int = variable number
	|	TypeCons !TypeCodeConstructor
	|	TypeApp !TypeCode !TypeCode
	|	_TypePatternVar !Int			// used internally between compiler and
										// unification algorithm
	|	_TypeEmpty						// used internally during unicifation

// macros to remove ugly underscores

TypePatternVar
	:==	_TypePatternVar
TypeEmpty
	:==	_TypeEmpty

// sanity check for closed types
checkClosedType :: TypeCode -> TypeCode
checkClosedType type_code
	| is_valid_type_scheme type_code
		=	type_code
	where
		is_valid_type_scheme :: TypeCode -> Bool
		// type schemes can only occur at the top level (rank 1)
		is_valid_type_scheme (TypeScheme n type)
			| n <= 0
				=	fatal "checkClosedType"
						("invalid number of type vars " +++  toString n)
			| [TypeVar i \\ i <- [0..n-1]] <> FV type
				=	fatal "checkClosedType" "invalid range of type vars"
			// otherwise
				=	is_valid_type type

		is_valid_type :: TypeCode -> Bool
		is_valid_type (TypeVar n)
			=	True
		is_valid_type (TypeCons _)
			=	True
		is_valid_type (TypeApp t a)
			=	is_valid_type t && is_valid_type a
		is_valid_type (_TypePatternVar n)
				=	fatal "checkClosedType"
						("unexpected _TypePatternVar " +++  toString n)
		is_valid_type _TypeEmpty
				=	fatal "checkClosedType" "unexpected _TypeEmpty"

concatStrings :: [{#Char}] -> .{#Char}
concatStrings l
	=	updateS 0 l (_createArray (sum [size s \\ s <- l]))
	where
		updateS :: !Int [{#Char}] *{#Char} -> *{#Char}
		updateS i [] s
			=	s
		updateS i [h : t] s
			=	updateS (i + size h) t {s & [pos] = c \\ c <-: h & pos <- [i..]}

// this only works for closed types
instance toString TypeCode where
	toString type
		=	concatStrings (show False type)
	where
		show :: Bool TypeCode -> [{#Char}]
		show pars (TypeScheme n type)
			=	parentheses pars (["A."] ++ showTypeVars [0 .. n-1] ++ [": " : show False type])
			where
				showTypeVars tvs
					=	sepBy " " (map showTypeVar tvs)
		show _ (TypeVar tv)
			=	[showTypeVar tv]
		show _ (TypeCons cons)
			=	[toString cons]
		show pars t
			=	show_app pars (reverse (uncurry_rev t))
			where
				uncurry_rev :: TypeCode -> [TypeCode]
				uncurry_rev (TypeApp t a)
					=	[a : uncurry_rev t]
				uncurry_rev t
					=	[t]

		show_app :: Bool [TypeCode] -> [{#Char}]
		show_app pars [TypeCons cons : args]
			| cons == TypeCodeConstructor_Arrow && length args == 2
				=	parentheses pars (show True (args !! 0)
							++ [" -> "] ++ show False (args !! 1))
			# (arity, pre, post)
				=	is_special cons
			| arity == length args
				=	[pre] ++ flatten (sepBy [", "] (map (show False) args)) ++ [post]
		show_app pars l
			= parentheses pars (flatten (sepBy [" "] (map (show True) l)))

		parentheses :: Bool [{#Char}] -> [{#Char}]
		parentheses pars l
			| pars
				=	["("] ++ l ++ [")"]
			// otherwise
				=	l

		is_special :: TypeCodeConstructor -> (!Int, !{#Char}, !{#Char})
		is_special cons
			| not (typeCodeConstructorIsPredefined cons)
			|| cons == TypeCodeConstructorInt
			|| cons == TypeCodeConstructorChar
			|| cons == TypeCodeConstructorReal
			|| cons == TypeCodeConstructorBool
			|| cons == TypeCodeConstructorDynamic
			|| cons == TypeCodeConstructorFile
			|| cons == TypeCodeConstructorWorld
			|| cons == TypeCodeConstructor_Arrow
				=	(-1, "", "")
			| cons == TypeCodeConstructor_List
				=	(1, "[", "]")
			| cons == TypeCodeConstructor_StrictList
				=	(1, "[!", "]")
			| cons == TypeCodeConstructor_UnboxedList
				=	(1, "[#", "!]")
			| cons == TypeCodeConstructor_TailStrictList
				=	(1, "[", "!]")
			| cons == TypeCodeConstructor_StrictTailStrictList
				=	(1, "[!", "!]")
			| cons == TypeCodeConstructor_UnboxedTailStrictList
				=	(1, "[#", "!]")
			| cons == TypeCodeConstructor_LazyArray
				=	(1, "{", "}")
			| cons == TypeCodeConstructor_StrictArray
				=	(1, "{!", "}")
			| cons == TypeCodeConstructor_UnboxedArray
				=	(1, "{#", "}")
			=	is_tuple_cons 2 32 cons
			where
				is_tuple_cons i n cons
					| i > n
						=	(-1, "", "")
					| cons == TypeCodeConstructor_Tuple i
						=	(i, "(", ")")
					// otherwise
						=	is_tuple_cons (i+1) n cons

sepBy :: a [a] -> [a]
sepBy _ []
	=	[]
sepBy _ [x]
	=	[x]
sepBy sep [h : t]
	=	[h, sep : sepBy sep t]

showTypeVar :: Int -> {#Char}
showTypeVar tv
	=	toString (toChar (toInt 'a' + tv))

instance == TypeCode where
	(==) (TypeScheme n1 t1) (TypeScheme n2 t2)
		=	n1 == n2 && t1 == t2
	(==) (TypeApp t1 arg1) (TypeApp t2 arg2)
		=	t1 == t2 && arg1 == arg2
	(==) (TypeVar a) (TypeVar b)
		=	a == b
	(==) (TypeCons a) (TypeCons b)
		=	a == b
	(==) (TypePatternVar a) (TypePatternVar b)
		=	a == b
	(==) TypeEmpty TypeEmpty
		=	True
	(==) _ _
		=	False

/*	----------------------------------------------------------------------------
	Unification functions
-----------------------------------------------------------------------------  */

:: _UnificationEnvironment
//	:==	UnifierStraightforward
	:==	UnifierEfficient
//	:==	UnifiersCombined

_initial_unification_environment :: !Int !Int -> *_UnificationEnvironment
_initial_unification_environment n_type_pattern_vars n_type_vars
	=	initial_unification_environment n_type_pattern_vars n_type_vars

_bind_global_type_pattern_var :: !TypeCode !T_ypeObjectType
		!*_UnificationEnvironment -> *_UnificationEnvironment
_bind_global_type_pattern_var var (UV_Placeholder type) subst
	=	bind_global_type_pattern_var var type subst
_bind_global_type_pattern_var _ _ _
	=	incorrect_encoding "_bind_global_type_pattern_var"

_unify :: !_UnificationEnvironment !T_ypeObjectType !TypeCode
									-> (!Bool, _UnificationEnvironment)
_unify subst (UV_Placeholder t1) t2
	=	unify_types subst t1 t2
_unify subst t _
	=	incorrect_encoding "_unify"

_normalise :: !Bool !_UnificationEnvironment !TypeCode -> T_ypeObjectType
_normalise norm subst t
	| norm
		=	UV_Placeholder (normalise subst t)
	// otherwise
		=	UV_Placeholder t

incorrect_encoding :: {#Char} -> .a
incorrect_encoding function_name
	=	fatal function_name ("the encoding of the dynamic type is incorrect"
				+++ "(probably a version conflict)")

fatal :: {#Char} {#Char} -> .a
fatal function_name message
	=	abort ("_SystemDynamic, " +++ function_name +++ ": " +++ message)

typeCodeOfDynamic :: !Dynamic -> TypeCode
typeCodeOfDynamic d
	// help the confused compiler
	=	typeOf (cast d)
	where
		typeOf :: DynamicTemp -> TypeCode
		typeOf {_type=UV_Placeholder type}
			=	type
		typeOf _
			=	incorrect_encoding "typeCodeOfDynamic"

		cast :: !a -> b
		cast _
			=	code
			{
				pop_a 0
			}

/*	----------------------------------------------------------------------------
	Interface to the unifier
-----------------------------------------------------------------------------  */

// dynamic unifier class

class DynamicUnifier a where
	initial_unification_environment :: !Int !Int -> *a
	bind_global_type_pattern_var :: !TypeCode !TypeCode !*a -> *a
	unify_types :: !a !TypeCode !TypeCode -> (!Bool, !a)
	normalise :: !a !TypeCode -> TypeCode


/*	----------------------------------------------------------------------------
	Combined unification

	Combine the efficient and the straightforward unifiers and compare the
	results. This combined unifier gives a fatal error message if two unifiers
	disagree.
-----------------------------------------------------------------------------  */

:: UnifiersCombined
	=	UnifiersCombined .UnifierStraightforward .UnifierEfficient

instance DynamicUnifier UnifiersCombined where
	initial_unification_environment n fixed
		=	UnifiersCombined
				(initial_unification_environment n fixed)
				(initial_unification_environment n fixed)

	bind_global_type_pattern_var tpv ts (UnifiersCombined u1 u2)
		=	UnifiersCombined
				(bind_global_type_pattern_var tpv ts u1)
				(bind_global_type_pattern_var tpv ts u2)

	unify_types (UnifiersCombined u1 u2) t1 t2
		# (b1, u1)
			=	unify_types u1 t1 t2
		# (b2, u2)
			=	unify_types u2 t1 t2
		=	(	compare ("unify_types (" +++ toString t1 +++ " & "
										+++ toString t2 +++ ")") b1 b2
			,	UnifiersCombined u1 u2
			)

	normalise (UnifiersCombined u1 u2) type
		# type1`
			=	normalise u1 type
		# type2`
			=	normalise u2 type
		=	compare "normalise" type1` type2`

compare :: {#Char} a a -> a | toString,== a
compare function a b
	| a <> b
		=	fatal function
				("disagreement "	+++ toString a +++ " <> " +++ toString b)
	// otherwise
		=	a

/*	----------------------------------------------------------------------------
	Efficient unification

	A reasonably efficient unifier.
-----------------------------------------------------------------------------  */

:: UnifierEfficient =
	{	subst	:: !.{!TypeCode}
	,	fixed	:: !Int
	,	global_tpvs :: ![Int]
	}

:: Substitutions :== UnifierEfficient

instance DynamicUnifier UnifierEfficient where
	initial_unification_environment n fixed
		=	{subst = createArray n TypeEmpty, fixed = ~fixed-1, global_tpvs = []}

	bind_global_type_pattern_var tpv ts=:(TypeScheme n _) subst_env=:{fixed}
		# (subst_env, type)
			=	fresh_type_efficient True subst_env ts
		=	bind_global_type_pattern_var tpv type subst_env
	bind_global_type_pattern_var (TypePatternVar n) type subst_env
		// sanity check ...
		| subst_env.subst.[n] <> TypeEmpty
			=	fatal "bind_global_type_pattern_var" "already bound"
		// ... sanity check
		=	{subst_env & subst.[n] = type, global_tpvs = [n : subst_env.global_tpvs]}

	unify_types subst t1 t2
		# (subst, t1)
			=	fresh_type_efficient False subst t1
		# (unified, subst)
			=	unify_efficient t1 t2 subst
		| unified
			| proper t2 subst
				=	(True, subst)
			// otherwise
				=	(False, subst)
		// otherwise
			=	(False, subst)

	normalise subst_env (TypeScheme _ type)
		=	normalise subst_env type
	normalise subst_env type
		# norm
			=	createArray (size subst_env.subst - subst_env.fixed + 1) TypeEmpty
		# (_, n, _, type)
			=	normalise_type subst_env 0 norm type
		| n > 0
			=	TypeScheme n type
		// otherwise
			=	type

class normalise_type a :: Substitutions Int *{!TypeCode} a -> (Bool, Int, *{!TypeCode}, a)

instance normalise_type [a] | normalise_type a where
	normalise_type _ n norm []
		=	(False, n, norm, [])
	normalise_type subst_env n norm l=:[h : t]
		# (rh, n, norm, h`)
			=	normalise_type subst_env n norm h
		# (rt, n, norm, t`)
			=	normalise_type subst_env n norm t
		| rh || rt
			=	(True, n, norm, [h` : t`])
		// otherwise
			=	(False, n, norm, l)

instance normalise_type TypeCode where
	normalise_type subst_env n norm (TypePatternVar i)
		# (n, norm, t)
			=	normalise_type_var subst_env n norm i
		=	(True, n, norm, t)
	normalise_type subst_env n norm (TypeVar i)
		# (n, norm, t)
			=	normalise_type_var subst_env n norm i
		=	(True, n, norm, t)
	normalise_type subst_env n norm tc=:(TypeCons _)
		=	(False, n, norm, tc)
	normalise_type subst_env n norm ta=:(TypeApp t arg)
		# (rt, n, norm, t`)
			=	normalise_type subst_env n norm t
		# (ra, n, norm, arg`)
			=	normalise_type subst_env n norm arg
		| rt || ra
			=	(True, n, norm, TypeApp t` arg`)
		// otherwise
			=	(False, n, norm, ta)

normalise_type_var :: Substitutions Int *{!TypeCode} Int -> (Int, *{!TypeCode}, TypeCode)
normalise_type_var subst_env n norm i
	# (ni, norm)
		=	type_var_number i norm
	# (ns, norm)
		=	norm![ni] 
	| ns == TypeEmpty
		// the substitution for this variable is not normalised yet
		# s
			=	if (i >= 0) subst_env.subst.[i] TypeEmpty
		| s == TypeEmpty
			// this is the first encounter of an unbound type pattern variable
			# ns
				=	TypeVar n
			=	(n+1, {norm & [ni] = ns}, ns)
		// otherwise
			# (_, n, norm, ns)
				=	normalise_type subst_env n norm s
			=	(n, {norm & [i] = ns}, ns)
	// otherwise
		// the substitution is already normalised		
		=	(n, norm, ns)
	where
		type_var_number :: Int *{!TypeCode} -> (Int, *{!TypeCode})
		type_var_number i norm
			| i >= 0
				=	(i, norm)
			// otherwise
				# (n, norm)
					=	usize norm
				=	(n + i, norm)

isIndirection :: TypeCode -> Bool
isIndirection TypeEmpty
	=	False
isIndirection type
	=	True

// Meister Proper == Mr. Clean (see <http://www.mrproper.de/>
// FIXME: this is a quick hack, try to think of something better
proper :: TypeCode Substitutions -> Bool
proper type subst_env
	# (_, p)
		=	binds_to_fixed type subst_env (createArray (size subst_env.subst) '?')
	=	length [1 \\ i <- [0..size p-1] | p.[i] == 'y' && not (isMember i subst_env.global_tpvs)] == 0

class binds_to_fixed a :: a Substitutions *{#Char} -> (!Bool, !*{#Char})

instance binds_to_fixed [a] | binds_to_fixed a where
	binds_to_fixed [] subst_env p
		=	(False, p)
	binds_to_fixed [t : ts] subst_env p
		# (fixed_t, p)
			=	binds_to_fixed t subst_env p
		# (fixed_ts, p)
			=	binds_to_fixed ts subst_env p
		=	(fixed_t || fixed_ts, p)

instance binds_to_fixed TypeCode where
	binds_to_fixed (TypePatternVar tv_number) subst_env p
		# (c, p)
			=	p![tv_number]
		| c == 'y'
			=	(True, p)
		| c == 'n'
			=	(False, p)
		| c == '?'
			# (fixed, p)
				=	binds_to_fixed subst_env.subst.[tv_number] subst_env p
			=	(fixed, {p & [tv_number] = if fixed 'y' 'n'})
	binds_to_fixed tv1=:(TypeVar tv_number) subst_env p
		| tv_number < 0
			=	(True, p)
		# (c, p)
			=	p![tv_number]
		| c == 'Y'
			=	(True, p)
		| c == 'N'
			=	(False, p)
		| c == '?'
			# (fixed, p)
				=	binds_to_fixed subst_env.subst.[tv_number] subst_env p
			# fixed
				=	is_fixed tv_number || fixed
			=	(fixed, {p & [tv_number] = if fixed 'Y' 'N'})
	binds_to_fixed (TypeCons _) subst_env p
		=	(False, p)
	binds_to_fixed (TypeApp t arg) subst_env p
		=	binds_to_fixed [t, arg] subst_env p
	binds_to_fixed TypeEmpty _ p
		=	(False, p)

// proper & occurs check
improper :: Int Int TypeCode -> Bool
improper fixed tv_number1 (TypeVar tv_number2)
	=	tv_number1 == tv_number2 || is_fixed tv_number2
improper fixed tv_number1 (TypePatternVar tv_number2)
	=	tv_number1 == tv_number2
improper fixed tv_number1 (TypeCons _)
	=	False
improper fixed tv_number (TypeApp t arg)
	=	any (improper fixed tv_number) [t, arg]
improper fixed tv_number TypeEmpty
	=	False
improper _ _ ts=:(TypeScheme _ _)
	=	fatal "improper" ("unexpected type scheme " +++ toString ts)

occurs :: Int TypeCode -> Bool
occurs tv_number1 (TypeVar tv_number2)
	=	tv_number1 == tv_number2
occurs tv_number1 (TypePatternVar tv_number2)
	=	tv_number1 == tv_number2
occurs tv_number1 (TypeCons _)
	=	False
occurs tv_number (TypeApp t arg)
	=	any (occurs tv_number) [t, arg]
occurs tv_number TypeEmpty
	=	False
occurs _ ts=:(TypeScheme _ _)
	=	fatal "occurs" ("unexpected type scheme " +++ toString ts)

is_fixed tv_number
	:==	tv_number < 0


class unify_efficient a :: a a *Substitutions -> (Bool, *Substitutions)

instance unify_efficient [a] | unify_efficient a where
	unify_efficient [] [] subst_env
		=	(True, subst_env)
	unify_efficient [t1 : t1s] [t2 : t2s] subst_env
		# (unified, subst_env)
			=	unify_efficient t1 t2 subst_env
		| unified
			=	unify_efficient t1s t2s subst_env
		// otherwise
			=	(False, subst_env)
	unify_efficient _ _ subst_env
		=	(False, subst_env)

bound_type tv_number subst_env
	| tv_number < 0
		=	( TypeEmpty, subst_env)
	// otherwise
		=	subst_env!subst.[tv_number]

instance unify_efficient TypeCode where
	unify_efficient tv1=:(TypePatternVar tv_number1) tv2=:(TypePatternVar tv_number2) subst_env
		# (t1, subst_env)
			=	subst_env!subst.[tv_number1]
		| isIndirection t1
			=	unify_efficient t1 tv2 subst_env
		# (t2, subst_env)
			=	subst_env!subst.[tv_number2]
		| isIndirection t2
			=	unify_efficient tv1 t2 subst_env
		| tv_number1 == tv_number2
			=	(True, subst_env)
		// otherwise
			=	(True, {subst_env & subst.[tv_number1] = tv2})
	unify_efficient (TypePatternVar tv_number1) t2 subst_env
		# (t1, subst_env)
			=	subst_env!subst.[tv_number1]
		| isIndirection t1
			=	unify_efficient t1 t2 subst_env
		| improper subst_env.fixed tv_number1 t2
			=	(False, subst_env)
		// otherwise
			=	(True, {subst_env & subst.[tv_number1] = t2})
	unify_efficient t1 (TypePatternVar tv_number2) subst_env
		# (t2, subst_env)
			=	subst_env!subst.[tv_number2]
		| isIndirection t2
			=	unify_efficient t1 t2 subst_env
		| improper subst_env.fixed tv_number2 t1
			=	(False, subst_env)
		// otherwise
			=	(True, {subst_env & subst.[tv_number2] = t1})
	unify_efficient tv1=:(TypeVar tv_number1) tv2=:(TypeVar tv_number2) subst_env
		# (t1, subst_env)
			=	bound_type tv_number1 subst_env
		| isIndirection t1
			=	unify_efficient t1 tv2 subst_env
		# (t2, subst_env)
			=	bound_type tv_number2 subst_env
		| isIndirection t2
			=	unify_efficient tv1 t2 subst_env
		| tv_number1 == tv_number2
			=	(True, subst_env)
		| not (is_fixed tv_number1 || occurs tv_number1 tv2)
			=	(True, {subst_env & subst.[tv_number1] = tv2})
		| not (is_fixed tv_number2 || occurs tv_number2 tv1)
			=	(True, {subst_env & subst.[tv_number2] = tv1})
		// otherwise
			=	(False, subst_env)
	unify_efficient tv1=:(TypeVar tv_number1) t2 subst_env
		# (t1, subst_env)
			=	bound_type tv_number1 subst_env
		| isIndirection t1
			=	unify_efficient t1 t2 subst_env
		| not (is_fixed tv_number1 || occurs tv_number1 t2)
			=	(True, {subst_env & subst.[tv_number1] = t2})
		// otherwise
			=	(False, subst_env)
	unify_efficient t1 tv2=:(TypeVar tv_number2) subst_env
		# (t2, subst_env)
			=	bound_type tv_number2 subst_env
		| isIndirection t2
			=	unify_efficient t1 t2 subst_env
		| not (is_fixed tv_number2 || occurs tv_number2 t1)
			=	(True, {subst_env & subst.[tv_number2] = t1})
		// otherwise
			=	(False, subst_env)
	unify_efficient (TypeCons cons1) (TypeCons cons2) subst_env
		| cons1 == cons2
			=	(True, subst_env)
		// otherwise
			=	(False, subst_env)
	unify_efficient (TypeApp t1 arg1) (TypeApp t2 arg2) subst_env
		# (unified_ts, subst_env)
			=	unify_efficient t1 t2 subst_env
		| unified_ts
			# (unified_args, subst_env)
				=	unify_efficient arg1 arg2 subst_env
			| unified_args
				=	(True, subst_env)
			// otherwise
				=	(False, subst_env)
		// otherwise
			=	(False, subst_env)
	unify_efficient _ _ subst_env
		=	(False, subst_env)


fresh_type_efficient :: Bool Substitutions TypeCode -> (*Substitutions, TypeCode)
fresh_type_efficient fix subst_env (TypeScheme n type)
	| fix
		# type
			=	renumber_type_vars subst_env.fixed type
		=	({copy_substitutions subst_env & fixed = subst_env.fixed - n}, type)
	// otherwise
		# type
			=	renumber_type_vars (size subst_env.subst) type
		=	(extend_substitutions n subst_env, type)
fresh_type_efficient _ subst_env type
	=	(copy_substitutions subst_env, type)

copy_substitutions :: Substitutions -> *Substitutions
copy_substitutions subst_env
	=	{subst_env & subst = {e \\ e <-: subst_env.subst}}

extend_substitutions :: Int Substitutions -> *Substitutions
extend_substitutions n subst_env
	=	{	subst_env
		&	subst
				=	{	createArray (size subst_env.subst + n) TypeEmpty
					&	[i] = s \\	s <-: subst_env.subst & i <- [0..]
					}
		}

renumber_type_vars :: Int TypeCode -> TypeCode
renumber_type_vars n t
	| n == 0 || not r
		=	t
	// otherwise
		=	t`
	where
		(r, t`)
			=	renumber_vars n t

class renumber_vars a :: Int a -> (Bool, a)

instance renumber_vars [a] | renumber_vars a where
	renumber_vars _ []
		=	(False, [])
	renumber_vars n l=:[h : t]
		# (rh, h`)
			=	renumber_vars n h
		# (rt, t`)
			=	renumber_vars n t
		| rh || rt
			=	(True, [h` : t`])
		// otherwise
			=	(False, l)

instance renumber_vars TypeCode where
	renumber_vars n (TypeVar i)
		=	(True, TypeVar (if (n > 0) (i+n) (n-i)))
	renumber_vars n tc=:(TypeCons _)
		=	(True, tc)
	renumber_vars n ta=:(TypeApp t arg)
		# (rt, t`)
			=	renumber_vars n t
		# (ra, arg`)
			=	renumber_vars n arg
		| rt || ra	
			=	(True, TypeApp t` arg`)
		// otherwise
			=	(False, ta)

/*	----------------------------------------------------------------------------
	Straightforward unification

	The unification algorithm from Marco Pil's thesis (chapter 4). This unifier
	serves as a reference implementation.
-----------------------------------------------------------------------------  */

:: UnifierStraightforward
	:==	UnifyEnv

:: UnifyEnv
	=	{	us_subst :: Substitution
		,	us_next_fresh_var :: Int
		,	us_fixed :: Int
		}

instance DynamicUnifier UnifierStraightforward where
	initial_unification_environment n fixed
		=	{us_subst = [], us_next_fresh_var = n, us_fixed = ~fixed-1}

	bind_global_type_pattern_var tpv t subst_env=:{us_subst, us_fixed}
		# (us_fixed, t)
			=	fresh_type False us_fixed t
		=	{	subst_env
			&	us_subst = [(tpv, t) : us_subst]
			,	us_fixed = us_fixed
			}

	// first type arg is the actual type, second is the type pattern
	unify_types subst_env=:{us_subst, us_next_fresh_var} t1 t2
		# (us_next_fresh_var, t1)
			=	fresh_type False us_next_fresh_var t1
		# t1
			=	applyS us_subst t1
		# t2
			=	applyS us_subst t2
		# fixed
			=	[TypeVar i \\ TypeVar i <- FV t2 | i < 0]
		=	case unify fixed t1 t2 of
				No
					-> (False, {subst_env & us_subst = undef})
				Yes subst1
					# subst_env
						=	{	subst_env
							&	us_subst = us_subst o` subst1
							,	us_next_fresh_var = us_next_fresh_var
							}
					->	(True, subst_env)

	normalise subst (TypeScheme _ type)
		=	normalise subst type
	normalise {us_subst} type
		# type
			=	foldl (flip applyS) type [[subst] \\ subst <- us_subst]
		=	close type

:: Set a
	:==	[a]

:: Maybe2 a
	=	No
	|	Yes a

:: Substitution
	:==	[(TypeCode, TypeCode)]

:: TypeDefinition
	:==	TypeCodeConstructor

fresh_type :: Bool Int TypeCode -> (Int, TypeCode)
fresh_type fresh_tpvs next_fresh_var type
	# type
		=	removeQuantifier type
	# vars
		=	FV type ++ if fresh_tpvs (TPV type) []
	# n_vars
		=	length vars
	| n_vars > 0
		# fresh_vars
			=	if (next_fresh_var < 0)
					[next_fresh_var, next_fresh_var-1 ..]
					[next_fresh_var ..]
		# type
			=	applyS (zip2 vars [TypeVar i \\ i <- fresh_vars]) type 
		# next_fresh_var
			=	if (next_fresh_var < 0)
					(next_fresh_var - n_vars)
					(next_fresh_var + n_vars)
		=	(next_fresh_var, type)
	// otherwise
		=	(next_fresh_var, type)

close :: TypeCode -> TypeCode
close type
	# type
		=	snd (fresh_type True 0 type)
	# n_vars
		=	length (FV type)
	| n_vars > 0
		=	TypeScheme n_vars type
	// otherwise
		=	type

removeQuantifier :: TypeCode -> TypeCode
removeQuantifier (TypeScheme _ type)
	=	removeQuantifier type
removeQuantifier type
	=	type

unify :: [TypeCode] TypeCode TypeCode -> Maybe2 Substitution
unify fix a=:(TypePatternVar _) b=:(TypePatternVar _)
	| a == b
		=	Yes unitSubstitution
	// otherwise
		=	Yes (substitute a b)
unify fix a=:(TypePatternVar _) sigma
	| FV sigma `disjoint` fix && a `not_in` TPV sigma
		=	Yes (substitute a sigma)
	// otherwise
		=	No
unify fix sigma a=:(TypePatternVar _)
	| FV sigma `disjoint` fix && a `not_in` TPV sigma
		=	Yes (substitute a sigma)
	// otherwise
		=	No
unify fix alpha=:(TypeVar _) beta=:(TypeVar _)
	| alpha == beta
		=	Yes unitSubstitution
	| alpha `not_in` fix
		=	Yes (substitute alpha beta)
	| beta `not_in` fix
		=	Yes (substitute beta alpha)
	// otherwise
		=	No
unify fix alpha=:(TypeVar _) sigma
	| alpha `not_in` fix && alpha `not_in` TPV sigma
		=	Yes (substitute alpha sigma)
	// otherwise
		=	No
unify fix sigma alpha=:(TypeVar _)
	| alpha `not_in` fix && alpha `not_in` TPV sigma
		=	Yes (substitute alpha sigma)
	// otherwise
		=	No
unify fix (TypeCons cons1) (TypeCons cons2)
	| cons1 == cons2
		=	Yes unitSubstitution
	// otherwise
		=	No
unify fix (TypeApp t1 arg1) (TypeApp t2 arg2)
	=	unifyArgs fix [t1, arg1] [t2, arg2]
unify fix _ _ 
	=	No

unifyArgs :: [TypeCode] [TypeCode] [TypeCode] -> Maybe2 Substitution
unifyArgs fix [] []
	=	Yes unitSubstitution
unifyArgs fix [] taus
	=	No
unifyArgs fix sigmas []
	=	No
unifyArgs fix [sigma:sigmas] [tau:taus]
	=	case unify fix sigma tau of
			Yes sub
				->	case unifyArgs fix
							(map (applyS sub) sigmas)
							(map (applyS sub) taus)
					of
						Yes subs
							| isProperS fix (subs o` sub)
								->	Yes (subs o` sub)
							// otherwise
								->	No
						No
							->	No
			No
				->	No

// Yields the identity substitution
unitSubstitution :: Substitution
unitSubstitution
	=	[]

// Yields the singleton substitution that assigns the second argument (a type)
//  to the first argument (a type variable)
substitute :: TypeCode TypeCode -> Substitution
substitute alpha sigma
	=	[(alpha, sigma)]

// Applies a substitution to a type
applyS :: Substitution TypeCode -> TypeCode
applyS subst alpha=:(TypeVar info)
	=	hd [type \\ (TypeVar var, type) <- subst ++ substitute alpha alpha | var == info]
applyS subst a=:(TypePatternVar info)
	=	hd [type \\ (TypePatternVar var, type) <- subst ++ substitute a a | var == info]
applyS subst tc=:(TypeCons _)
	=	tc
applyS subst (TypeApp t arg)
	=	TypeApp (applyS subst t) (applyS subst arg)

// Checks whether the substitution is proper w.r.t. a set of
//  generic type variables (see definition 3.4.4)
isProperS :: [TypeCode] Substitution -> Bool
isProperS fix subst
	=	and [fix `disjoint` FV (applyS subst a) \\ a =:(TypePatternVar _) <- sup subst]

sup :: Substitution -> [TypeCode]
sup subst
	=	[var \\ (var, _) <- subst]

// Yields the composition of two substitutions
(o`) infix :: Substitution Substitution -> Substitution
(o`) subst1 subst2
	// sanity check ...
	| not (sup subst1 `disjoint` sup subst2)
		=	fatal "o`" "illegal substitutions"
	// ... sanity check
	=	[(tv1, applyS subst2 type1) \\ (tv1, type1) <- subst1] ++ subst2

// Yields the set of all free generic type variables in a type
//  (see also definition 3.2.2)
FV :: TypeCode -> Set TypeCode
FV alpha=:(TypeVar _)
	=	[alpha]
FV (TypePatternVar _)
	=	[]
FV (TypeCons _)
	=	[]
FV (TypeApp t arg)
	=	U [FV t, FV arg]

// Yields the set of all type pattern variables in a type
//  (see also definition 3.2.3)
TPV :: TypeCode -> Set TypeCode
TPV (TypeVar _)
	=	[]
TPV a=:(TypePatternVar _)
	=	[a]
TPV (TypeCons _)
	=	[]
TPV (TypeApp t arg)
	=	U [TPV t, TPV arg]

// yields true as the first argument does not occur in the second,
//  and false otherwise
(`not_in`) infix :: a (Set a) -> Bool | Eq a
(`not_in`) element set
	=	not (isMember element set)

// yields true as no element appears in both sets,
//  and false otherwise (See also definition 3.2.7)
(`disjoint`) infix :: (Set a) (Set a) -> Bool | Eq a
(`disjoint`) set1 set2
	=	not (isAnyMember set1 set2)

// Union over a list of sets
U :: ([Set a] -> Set a) | Eq a
U
	=	removeDup o flatten

/*	----------------------------------------------------------------------------
	TypeCodeConstructor
	
	Each type is represented at run-time by a unique data constructor. The
	compiler generates an extra type for each user defined type. For example,
	the type
	
		:: List a = Nil | Cons (List a)

	is accompanied by the type

		:: TC;List a = TC;List (List a)

	This module contains similar definitions for Clean's predefined types.

	Two implementations of TC;Type are equivalent only if the implementations
	of Type are equivalent. The dynamic linker will use the same descriptor
	for constructors of equivalent types, so comparing to type constructors
	can now be done by comparing descriptors.
-----------------------------------------------------------------------------  */

:: TypeCodeConstructor
	=	TypeCodeConstructor // represents all type code constructors
:: TC_Int
	=	TC_Int
:: TC_Char
	=	TC_Char
:: TC_Real
	=	TC_Real
:: TC_Bool
	=	TC_Bool
:: TC_Dynamic
	=	TC_Dynamic
:: TC_File
	=	TC_File
:: TC_World
	=	TC_World
:: TC__Arrow
	=	TC__Arrow
:: TC__List a
	=	TC__List
:: TC__StrictList
	=	TC__StrictList
:: TC__UnboxedList
	=	TC__UnboxedList
:: TC__TailStrictList
	=	TC__TailStrictList
:: TC__StrictTailStrictList
	=	TC__StrictTailStrictList
:: TC__UnboxedTailStrictList
	=	TC__UnboxedTailStrictList
:: TC__Tuple2
	=	TC__Tuple2
:: TC__Tuple3
	=	TC__Tuple3
:: TC__Tuple4
	=	TC__Tuple4
:: TC__Tuple5
	=	TC__Tuple5
:: TC__Tuple6
	=	TC__Tuple6
:: TC__Tuple7
	=	TC__Tuple7
:: TC__Tuple8
	=	TC__Tuple8
:: TC__Tuple9
	=	TC__Tuple9
:: TC__Tuple10
	=	TC__Tuple10
:: TC__Tuple11
	=	TC__Tuple11
:: TC__Tuple12
	=	TC__Tuple12
:: TC__Tuple13
	=	TC__Tuple13
:: TC__Tuple14
	=	TC__Tuple14
:: TC__Tuple15
	=	TC__Tuple15
:: TC__Tuple16
	=	TC__Tuple16
:: TC__Tuple17
	=	TC__Tuple17
:: TC__Tuple18
	=	TC__Tuple18
:: TC__Tuple19
	=	TC__Tuple19
:: TC__Tuple20
	=	TC__Tuple20
:: TC__Tuple21
	=	TC__Tuple21
:: TC__Tuple22
	=	TC__Tuple22
:: TC__Tuple23
	=	TC__Tuple23
:: TC__Tuple24
	=	TC__Tuple24
:: TC__Tuple25
	=	TC__Tuple25
:: TC__Tuple26
	=	TC__Tuple26
:: TC__Tuple27
	=	TC__Tuple27
:: TC__Tuple28
	=	TC__Tuple28
:: TC__Tuple29
	=	TC__Tuple29
:: TC__Tuple30
	=	TC__Tuple30
:: TC__Tuple31
	=	TC__Tuple31
:: TC__Tuple32
	=	TC__Tuple32
:: TC__LazyArray
	=	TC__LazyArray
:: TC__StrictArray
	=	TC__StrictArray
:: TC__UnboxedArray
	=	TC__UnboxedArray


// compare two type code constructors: compare their descriptors
instance == TypeCodeConstructor where
	(==) _ _
		=	code inline
		{
			pushD_a 1
			pushD_a 0
			pop_a 2
			eqI
		}

// test for predefined type constructor, use the name of the type to
// recognise predefined types (otherwise we'd have to check for all
// predefined constructors)
typeCodeConstructorIsPredefined :: !TypeCodeConstructor -> Bool
typeCodeConstructorIsPredefined tc
		=	prefix tc == "TC_"
		where
			prefix :: !TypeCodeConstructor -> {#Char}
			prefix _
				=	code inline
					{
						pushD_a 0
					.d 0 1 i
						jsr DtoAC
					.o 1 0
						update_a 0 1
						pop_a	1
						pushI 2
						pushI 0
					.d 1 2 ii
						jsr sliceAC
					.o 1 0			
					}

// convert a type code constructor to a string: get its name and strip the
// first three characters (either "TC;" for user defined types or "TC_"
// for predefined types)
instance toString TypeCodeConstructor where
	toString _
		=	code
		{
			pushD_a 0
		.d 0 1 i
			jsr DtoAC
		.o 1 0
			update_a 0 1
			push_arraysize CHAR 0 1
			decI
			pushI 3
		.d 1 2 ii
			jsr sliceAC
		.o 1 0
		}

// cast to TypeCodeConstructor
toTypeCodeContructor :: !a -> TypeCodeConstructor
toTypeCodeContructor _
	=	code inline
	{
		update_a 0 1
		pop_a 1
	}

getTupleTypeConstructor arity
	:==	tupleTypeConstructors.[arity - 2]

tupleTypeConstructors :: {!TypeCodeConstructor}
tupleTypeConstructors
	=:
	{	toTypeCodeContructor TC__Tuple2
	,	toTypeCodeContructor TC__Tuple3
	,	toTypeCodeContructor TC__Tuple4
	,	toTypeCodeContructor TC__Tuple5
	,	toTypeCodeContructor TC__Tuple6
	,	toTypeCodeContructor TC__Tuple7
	,	toTypeCodeContructor TC__Tuple8
	,	toTypeCodeContructor TC__Tuple9
	,	toTypeCodeContructor TC__Tuple10
	,	toTypeCodeContructor TC__Tuple11
	,	toTypeCodeContructor TC__Tuple12
	,	toTypeCodeContructor TC__Tuple13
	,	toTypeCodeContructor TC__Tuple14
	,	toTypeCodeContructor TC__Tuple15
	,	toTypeCodeContructor TC__Tuple16
	,	toTypeCodeContructor TC__Tuple17
	,	toTypeCodeContructor TC__Tuple18
	,	toTypeCodeContructor TC__Tuple19
	,	toTypeCodeContructor TC__Tuple20
	,	toTypeCodeContructor TC__Tuple21
	,	toTypeCodeContructor TC__Tuple22
	,	toTypeCodeContructor TC__Tuple23
	,	toTypeCodeContructor TC__Tuple24
	,	toTypeCodeContructor TC__Tuple25
	,	toTypeCodeContructor TC__Tuple26
	,	toTypeCodeContructor TC__Tuple27
	,	toTypeCodeContructor TC__Tuple28
	,	toTypeCodeContructor TC__Tuple29
	,	toTypeCodeContructor TC__Tuple30
	,	toTypeCodeContructor TC__Tuple31
	,	toTypeCodeContructor TC__Tuple32
	}

TypeCodeConstructorInt :: TypeCodeConstructor
TypeCodeConstructorInt
	=	toTypeCodeContructor TC_Int

TypeCodeConstructorChar :: TypeCodeConstructor
TypeCodeConstructorChar
	=	toTypeCodeContructor TC_Char

TypeCodeConstructorReal :: TypeCodeConstructor
TypeCodeConstructorReal
	=	toTypeCodeContructor TC_Real

TypeCodeConstructorBool :: TypeCodeConstructor
TypeCodeConstructorBool
	=	toTypeCodeContructor TC_Bool

TypeCodeConstructorDynamic :: TypeCodeConstructor
TypeCodeConstructorDynamic
	=	toTypeCodeContructor TC_Dynamic

TypeCodeConstructorFile :: TypeCodeConstructor
TypeCodeConstructorFile
	=	toTypeCodeContructor TC_File

TypeCodeConstructorWorld :: TypeCodeConstructor
TypeCodeConstructorWorld
	=	toTypeCodeContructor TC_World

TypeCodeConstructor_Arrow :: TypeCodeConstructor
TypeCodeConstructor_Arrow
	=	toTypeCodeContructor TC__Arrow

TypeCodeConstructor_List :: TypeCodeConstructor
TypeCodeConstructor_List
	=	toTypeCodeContructor TC__List

TypeCodeConstructor_StrictList :: TypeCodeConstructor
TypeCodeConstructor_StrictList
	=	toTypeCodeContructor TC__StrictList

TypeCodeConstructor_UnboxedList :: TypeCodeConstructor
TypeCodeConstructor_UnboxedList
	=	toTypeCodeContructor TC__UnboxedList

TypeCodeConstructor_TailStrictList :: TypeCodeConstructor
TypeCodeConstructor_TailStrictList
	=	toTypeCodeContructor TC__TailStrictList

TypeCodeConstructor_StrictTailStrictList :: TypeCodeConstructor
TypeCodeConstructor_StrictTailStrictList
	=	toTypeCodeContructor TC__StrictTailStrictList

TypeCodeConstructor_UnboxedTailStrictList :: TypeCodeConstructor
TypeCodeConstructor_UnboxedTailStrictList
	=	toTypeCodeContructor TC__UnboxedTailStrictList

TypeCodeConstructor_Tuple :: !Int -> TypeCodeConstructor
TypeCodeConstructor_Tuple arity
	| arity < 2 || arity > 32
		=	fatal "TypeCodeConstructor_Tuple"
					("illegal tuple arity (" +++ toString arity +++ ")")
	// otherwise
		=	getTupleTypeConstructor arity

TypeCodeConstructor_LazyArray :: TypeCodeConstructor
TypeCodeConstructor_LazyArray
	=	toTypeCodeContructor TC__LazyArray

TypeCodeConstructor_StrictArray :: TypeCodeConstructor
TypeCodeConstructor_StrictArray
	=	toTypeCodeContructor TC__StrictArray

TypeCodeConstructor_UnboxedArray :: TypeCodeConstructor
TypeCodeConstructor_UnboxedArray
	=	toTypeCodeContructor TC__UnboxedArray


/* -----------------------------------------------------------------------------
  The rest of this module contains stuff for the dynamic linker. This should be
  moved to a separate module.
----------------------------------------------------------------------------- */

/* */

import StdFile
import StdEnv
from StdBool import not
import StdArray
import StdDynamicTypes

// import RWSDebug

import StdDynamicLowLevelInterface
import DynamicLinkerInterface
import DynamicUtilities

// --------------------------------------------------------------------------------------------------------------------------			
init_dynamic :: !String !DynamicHeader !*a -> *(.Bool,.GlobalDynamicInfoDummy,*a) | BinaryDynamicIO a
init_dynamic file_name dynamic_header=:{block_table_i,graph_i} file
	// a block table
	#! (ok,block_table,file)
		= read_block_table_from_dynamic dynamic_header file
	#! n_blocks
		= size block_table
	| not ok
		= (False,undef,file)
		
	// read graph blocks
	#! (ok,graph_blocks,file)
		= read_graph_blocks 0 n_blocks (createArray n_blocks "hallo") block_table dynamic_header file
	| not ok
		= (False,undef,file)
		
	// create global dynamic info
	#! gdi
		= { GlobalDynamicInfo |
			file_name		= file_name
		,	first_time		= True	
			
		,	block_table		= block_table
		,	id				= 0
		,	graph_blocks	= graph_blocks
		,	graph_pointers	= { {} \\ i <- [1..n_blocks] }
		
		,	diskid_to_runtimeid = {}
		,	di_disk_to_rt_dynamic_indices	= {}
		
		, 	di_dummy		= {}
		,	di_type_redirection_table		= {}

		}
	= (True,{gdid=gdi},file)
where 
	read_graph_blocks :: !Int !Int !*{String} !BlockTable !DynamicHeader !*f -> (!Bool,!{String},!*f) | BinaryDynamicIO f
	read_graph_blocks block_i n_blocks graph_blocks block_table dynamic_header=:{graph_i} file
		| block_i == n_blocks
			= (True,graph_blocks,file)

		#! block
			= block_table.[block_i]

		// read graph block
		#! (ok,file)
			= bd_seek file (block.bk_offset + graph_i) FSeekSet
		#! (graph_block,file)
			= bd_reads file block.bk_size
		| not ok || (size graph_block <> block.bk_size)
			= abort "read_graph_block: dynamic is corrupt"
		= read_graph_blocks (inc block_i) n_blocks {graph_blocks & [block_i] = graph_block} block_table dynamic_header file


// --------------------------------------------------------------------------------------------------------------------------			
:: *EncodedDynamic
	= { 
		ed_encoded_graph	:: !*{#Char}
	,	ed_dynamic_rts_info	:: !*{#Char}
	}
	
class EncodedDynamic a
where 
	dynamic_to_string :: !Dynamic -> (!Bool,!*a)

instance EncodedDynamic EncodedDynamic
where
	dynamic_to_string d
		#! d1 = d
		# (copy_graph_to_string,{ggtsf_o_n_library_instances=n_library_instances,ggtsf_o_range_table=range_table})
			= GetGraphToStringFunction
			
		# type_table_usage
			= (NF (createArray n_library_instances TTUT_UNUSED))	// indexed by RunTimeID or indirectly by converting a ModuleID to a RunTimeID

		#! cgtsa
			= { 
				cgtsa_dynamic					= d
			,	cgtsa_code_library_instances	= {} //createArray n_library_instances TTUT_UNUSED
			,	cgtsa_type_library_instances	= createArray n_library_instances TTUT_UNUSED
			,	cgtsa_range_table				= range_table
			}
		
		#! copy_graph_to_string_argument = /* NF */ {wrap_info = cgtsa}
		#! ({wrap_info = {cgtsr_encoded_dynamic,cgtsr_type_library_instances/*,cgtsr_code_library_instances*/,cgtsr_lazy_dynamic_references,cgtsr_runtime_ids}})
			= copy_graph_to_string copy_graph_to_string_argument
		
		#! gdri
			= {
				gdri_i_type_library_instances		= cgtsr_type_library_instances
			,	gdri_i_lazy_dynamics_references		= cgtsr_lazy_dynamic_references
			,	gdri_i_runtime_ids					= cgtsr_runtime_ids
			}
		#! dynamic_rts_info
			= GetDynamicRTSInfo	gdri

		# encoded_dynamic1
			= {
				ed_encoded_graph	= cgtsr_encoded_dynamic
			,	ed_dynamic_rts_info	= dynamic_rts_info
			}
		= (True,encoded_dynamic1)
	where 
		encode_type_table_usage :: !*{#Int} -> *{#Char}
		encode_type_table_usage type_table_usage
			#! encoded_type_table_usage
				= createArray (s_type_table_usage << 2) ' '
			= encode_type_table_usage 0 0 encoded_type_table_usage
		where
			encode_type_table_usage i offset encoded_type_table_usage
				| i == s_type_table_usage
					= encoded_type_table_usage
					
					# encoded_type_table_usage
						= WriteLong encoded_type_table_usage offset type_table_usage.[i]
					= encode_type_table_usage (inc i) (offset + 4) encoded_type_table_usage
		
			s_type_table_usage
				= size type_table_usage
		
instance EncodedDynamic String
where
	dynamic_to_string d
		# (ok,{ed_encoded_graph,ed_dynamic_rts_info})
			= dynamic_to_string d
			
		// size of arrays
		# (s_ed_encoded_graph,ed_encoded_graph)
			= usize ed_encoded_graph
		# (s_ed_dynamic_rts_info,ed_dynamic_rts_info)
			= usize ed_dynamic_rts_info
		# s_encoded_dynamic
			= s_ed_encoded_graph + s_ed_dynamic_rts_info
			
		// copy
		# (j,encoded_dynamic)
			= copy 0 s_ed_encoded_graph ed_encoded_graph 0 (createArray s_encoded_dynamic ' ')
			
		# (_,encoded_dynamic)
			= copy 0 s_ed_dynamic_rts_info ed_dynamic_rts_info j encoded_dynamic
			
		// patch encoded dynamic
		# encoded_dynamic
			= WriteLong encoded_dynamic (DYNAMIC_RTS_INFO_OFFSET - HEADER_SIZE_OFFSET) s_ed_encoded_graph
		# encoded_dynamic
			= WriteLong encoded_dynamic (DYNAMIC_RTS_INFO_SIZE - HEADER_SIZE_OFFSET) s_ed_dynamic_rts_info
		= (ok,encoded_dynamic)
	where 
		copy :: !Int !Int !{#Char} !Int !*{#Char} -> (!Int,!*{#Char})
		copy i limit src j dest
			| i == limit
				= (j,dest)
			= copy (inc i) limit src (inc j) {dest & [j] = src.[i]}

string_to_dynamic :: !String -> (!Bool,!Dynamic)
string_to_dynamic dynamic_as_string
	#! (odtl=:{odtl_o_ok})
		= OpenDynamicToLinker dynamic_as_string
	| not odtl_o_ok
		= abort "string_to_dynamic: error communicating with linker"

	# (file,{odtl_o_dynamic_rts_string=dynamic_rts_string})
		= extract_file_from_odtl odtl

	# (ok,dynamic_header,file)
		= read_dynamic_header file
	| not ok
		= abort "readDynamic"
	
	#! (ok,gdid,file)
		= init_dynamic "string_to_dynamic" dynamic_header file
	| not ok
		= abort "_SystemDynamic; internal error"

	#! dyn
//		= build_dynamic make_start_node_index { gdid & gdid.di_dummy = dynamic_rts_string}
		= build_block (NF make_start_node_index) (NF { gdid & gdid.di_dummy = dynamic_rts_string})


	| CloseDynamicToLinker {odtl & odtl_o_file = file}
	= (ok,dyn)
where
	extract_file_from_odtl odtl=:{odtl_o_file}
		= (odtl_o_file,{odtl & odtl_o_file = (default_elem,default_elemU)})
	
// --------------------------------------------------------------------------------------------------------------------------			
:: Wrap a 
	= { 
		wrap_info		:: !a
	}

// aanpassen van gesharde type door alle library instanties		
:: *CopyGraphToStringArguments
	= {
		cgtsa_dynamic					:: Dynamic
	,	cgtsa_code_library_instances	:: !*{#Int}						// unused
	,	cgtsa_type_library_instances	:: !*{#Int}
	,	cgtsa_range_table				:: !{#Char}
	}
	
:: *CopyGraphToStringResults
	= {
		cgtsr_encoded_dynamic			:: !*{#Char}
	,	cgtsr_code_library_instances	:: !*{#T_ypeConsSymbolInfo}		// unused
	,	cgtsr_type_library_instances	:: !*{#Int}
	,	cgtsr_lazy_dynamic_references	:: !{#LazyDynamicReference}
	,	cgtsr_runtime_ids				:: !{#RunTimeIDW}
	}
	
:: T_ypeConsSymbolInfo
	= {
		tcsi_type_and_module_name		:: !String
	,	tcsi_rt_library_instance		:: !Int
	}
		
// --------------------------------------------------------------------------------------------------------------------------				
/*
** the dynamic to be decoded is identified by (encoded_graph_i,graph)
**
**
** ugid				= returned by the dynamic rts on readDynamic's behalf.
** ulid				= local id of dynamic (within ugid; 0 for a top-level dynamic)
** encoded_graph_i	= index of dynamic to be decoded (0 for a top-level dynamic)
** graph			= string encoding of the *complete* dynamic to be decoded
*/

// change also string_to_graph.c; gts_gdi.c
:: GlobalDynamicInfo = {
	// general
		file_name						:: !String
	,	first_time						:: !Bool

	// block table 
	,	id								:: !Int				// id from Dynamic Linker
	,	block_table						:: !BlockTable		
	,	graph_blocks					:: !{String}		// filepointer to start of graph
	,	graph_pointers					:: !{#.{Int}}
	
	// 
	,	diskid_to_runtimeid				:: !{#Int}			// conversion from DiskId (disguished as RunTimeId) to *real* runtimeID (library instances)
	,	di_disk_to_rt_dynamic_indices	:: !{#Int} 			// conversion from disk to runtime index for lazy dynamics
	,	di_dummy						:: !String
	,	di_type_redirection_table		:: !{#RunTimeIDW}
	}
// The # above ensure that no ARRAY node is inserted.
	
// to prevent the unboxing of GlobalDynamicInfo
:: GlobalDynamicInfoDummy = {
		gdid			:: !GlobalDynamicInfo
	}

is_block_i_already_present block_i gdid=:{gdid={graph_pointers}}
	= size graph_pointers.[block_i] <> 0

:: Pointer = Pointer

GetBlockAddresses node_index gdid=:{gdid={id,first_time,file_name,block_table,graph_blocks,di_dummy=dynamic_rts_string}}
	#! gba_in
		= { default_elem &
			gba_i_filename				= file_name
		,	gba_i_first_time			= first_time
		,	gba_i_id					= id
		,	gba_i_block_i				= block_i
		,	gba_i_dynamic_rts_string	= dynamic_rts_string
		}
		
	#! (copy_string_to_graph,{gba_o_diskid_to_runtimeid,gba_o_disk_to_rt_dynamic_indices,gba_o_id,gba_o_addresses,gba_o_rt_type_redirection_table})
		= GetBlockAddresses2 gba_in;
	#! gdid
		= case first_time of
			True
				#! gdid
					= { gdid &
						gdid	=  { gdid.gdid &
									first_time						= False
								,	id								= gba_o_id
								,	diskid_to_runtimeid 			= gba_o_diskid_to_runtimeid
								,	di_disk_to_rt_dynamic_indices	= gba_o_disk_to_rt_dynamic_indices
								,	di_type_redirection_table		= gba_o_rt_type_redirection_table
								}
					}
				-> gdid
			False
				-> gdid

	= (copy_string_to_graph,gba_o_addresses,gdid)
where
	block_i
		= get_block_i node_index

/*
build_dynamic :: !Int !GlobalDynamicInfoDummy -> a //Dynamic
build_dynamic node_index gdid=:{gdid={id,first_time,file_name,block_table,graph_blocks,di_dummy=dynamic_rts_string,di_type_redirection_table}} //dynamic_rts_string
	#! (copy_string_to_graph,s_adr,gdid)
		= GetBlockAddresses node_index gdid;

	#! graph_block
		= graph_blocks.[block_i]

	# bk_entries
		= if (size (block_table.[block_i].bk_entries) == 0) 
			{block_table.[block_i].bk_offset - block_table.[block_i].bk_offset}	// and block_table.[block_i].bk_n_node_entries == 0
			(to_help_the_type_checker { en_offset - block_table.[block_i].bk_offset \\ en_offset <-: block_table.[block_i].bk_entries })		

	#! (graph2,_)
		= copy_string_to_graph 
			(s_adr % (8,size s_adr)) 			// %edx
			0									// %ebx offset in graph_block 
			graph_block							// %ecx graph
			gdid								// -4(%esi) unboxed GlobalDynamicInfo
			bk_entries 							// -8(%esi)
			block_i 							// %eax
			en_node_i							// (%esp)
		
		// netter zou zijn om de graph_pointers uniek te maken
	= graph2
where
	block_i
		= 0
	en_node_i
		= 0
	block
		= block_table.[block_i]
	
	s 
		= ">> block_i: " +++ toString block_i +++ "  en_node_i:" +++ toString en_node_i
*/
// calling convention:
// must be in a strict context and its arguments must have been evaluated completely using NF.	
build_block :: !Int !GlobalDynamicInfoDummy -> a //Pointer
build_block node_index gdid=:{gdid={id,first_time,file_name,block_table,graph_blocks,graph_pointers}}
//	| True <<- ("build_block", block_i)
//		= undef;
//	| first_time //<<- ("build_block", block_i)
//		= abort "dfkkdfk";
	| is_external_entry_node node_index 
		= abort ("build_block: internal error" +++ toString node_index)
	| not first_time && is_block_i_already_present block_i gdid // <<- ("block_i", block_i, "en-node", en_node_i, "size", size graph_pointers.[block_i])
	
	
	// is_block_i_already_present block_i gdid=:{gdid={graph_pointers}}
	// 	= size graph_pointers.[block_i] <> 0
 
		// blocku has already been constructed
		// There are multiple references from some decoded i.e. built block to some unbuilt 
		// i.e. undecoded block. If one of these references is evaluated, then the undecoded
		// block will be constructed.		
		= extract_already_built_graph block_i en_node_i gdid

	#! (copy_string_to_graph,s_adr,gdid)
		= GetBlockAddresses node_index gdid;
//	| COLLECT_AND_RENUMBER_EXTERNAL_TYPE_REFERENCES (size gdid.gdid.GlobalDynamicInfo.di_type_redirection_table <= 0) False
//		= abort ("build_dynamic; di_type_redirection_table should contain entries because otherwise the dynamic would be untyped")
		

	// fetch graph (semantics problem: not referentially transparent if dynamics are to be overwritten)
	// In the near future blocks may be read lazily from disk. This creates other problems because the
	// dynamic cannot be overwritten if there are still references to its encoded graph. Also if copied
	// dynamics are supported, then it is the same problem.
	#! graph_block
		= graph_blocks.[block_i]
//	|  H ("@building block " +++ toString block_i) True
	
	// build graph
	// - for the time being it assumed that each block only has one entry
	//   node
	// - should create a unique array containg pointers to already built blocks. 
	// en-nodes

	// copy_string_to_graph
	// input:
	// - entry node needed
	// - offsets to be update din the the graph_pointers table
	
	// output:
	// - graph belong to entry node
	// (- destructively updated graph_pointers table)
	# bk_entries
		= if (size (block_table.[block_i].bk_entries) == 0) 
			{block_table.[block_i].bk_offset - block_table.[block_i].bk_offset}	// and block_table.[block_i].bk_n_node_entries == 0
			(to_help_the_type_checker { en_offset - block_table.[block_i].bk_offset \\ en_offset <-: block_table.[block_i].bk_entries })		

	# (graph2,_)
		= copy_string_to_graph 
			(s_adr % (8,size s_adr)) 			// %edx
			0									// %ebx offset in graph_block 
			graph_block							// %ecx graph
			gdid								// -4(%esi) unboxed GlobalDynamicInfo
			bk_entries 							// -8(%esi)
			block_i 							// %eax
			en_node_i							// (%esp)

		// netter zou zijn om de graph_pointers uniek te maken
	= graph2
where
	block_i
		= get_block_i node_index
	en_node_i
		= get_en_node_i node_index
	block
		= block_table.[block_i]
	
	s 
		= ">> block_i: " +++ toString block_i +++ "  en_node_i:" +++ toString en_node_i
		
	// Ideas:
	// - If there are no references more to a particular block and not all blocks have been built, then there is
	//   a space leak because the graph_pointers-array still contains pointers to at least one entry node.
	// - Version information per block instead per dynamic. In case of a copied dynamic i.e. a reference to a
	//   piece of graph in another dynamic, another version of the string_to_graph-routine might have been used.
	//   This version of the conversion routine should then be called.
	// - Reading blocks lazily. The dynamic run-time system must guarantee that the dynamic from which the blocks
	//   are lazily read remains available until there are no references to that dynamic or all blocks have been
	//   read from the dynamic.
	// - If during building a block some external node i.e. in an another block is referenced more than once, then
	//   the build_block-closure is built multiple times. This could be optimized.
	// - Can internal entry nodes occur in absence of existential types?
	//
	// The function below with its local function should *not* call the garbage collector because the pointer to
	// an already existing piece of graph can change because of gc.
	extract_already_built_graph :: !Int !Int !GlobalDynamicInfoDummy -> a //Pointer
	extract_already_built_graph block_i en_node_i gdid=:{gdid={graph_pointers}}
		#! (p,graph_pointers) 
			= graph_pointers![block_i,en_node_i]
		# (g,_) 
			= convert_to_dynamic p
		= g
	where
		convert_to_dynamic :: Int -> (a /*Pointer*/,!Int)
		convert_to_dynamic _
			= code {
					pushI	0
			}

build_lazy_block :: !Int !Int -> a //Pointer
build_lazy_block node_index lazy_dynamic_index
	| is_internal_reference node_index
		= abort "build_lazy_block; internal error; internal reference"
	#! debug_string
		= "external reference to block " +++ toString (get_block_i node_index) +++ " with entry node " +++ toString (get_en_node_i node_index) +++ " with lazy dynamic index " +++ (toString lazy_dynamic_index)

	#! rld_o
		= RegisterLazyDynamic lazy_dynamic_index;
	#! (rld_o_file,{rld_o_filename,rld_o_diskid_to_runtimeid,rld_o_disk_to_rt_dynamic_indices,rld_o_id,rld_o_rt_type_redirection_table})
		= extract_rld_o_file rld_o;

	# (ok,dynamic_header,rld_o_file)
		= read_dynamic_header rld_o_file
		
	# (ok,gdid,rld_o_file)
		= init_dynamic rld_o_filename dynamic_header rld_o_file
	| not ok
		= abort "init_dynamic: init_dynamic failed"

	#! gdid
		= { gdid &
			gdid	=  { gdid.gdid &
						first_time						= False
					,	id								= rld_o_id
					,	diskid_to_runtimeid 			= rld_o_diskid_to_runtimeid
					,	di_disk_to_rt_dynamic_indices	= rld_o_disk_to_rt_dynamic_indices
					,	di_type_redirection_table		= rld_o_rt_type_redirection_table
					}
		}
	= build_block (NF node_index) (NF gdid)
where
	extract_rld_o_file rld_o=:{rld_o_file}
		= (rld_o_file,{rld_o & rld_o_file = (0,default_elemU)})


copy_graph_to_string_OK :: !(Wrap CopyGraphToStringArguments) -> (Wrap CopyGraphToStringResults)
copy_graph_to_string_OK _ 
	= {wrap_info={cgtsr_encoded_dynamic={},cgtsr_code_library_instances={},cgtsr_type_library_instances={},cgtsr_lazy_dynamic_references={},cgtsr_runtime_ids={}}}

to_help_the_type_checker i :== to_help_the_type_checker2 i
where
	to_help_the_type_checker2 :: !{#Int} -> {#Int}
	to_help_the_type_checker2 i
		= i	
/* */